#include <ESP8266WiFi.h>
#include <ESP8266mDNS.h>
#include <WiFiUdp.h>
#include <ArduinoOTA.h>
#include <SoftwareSerial.h>
 
const char* ssid = "SSID";  // Wi-Fi SSID
const char* password = "PASSWORD";  // Wi-Fi password
 
char thingSpeakAddress[] = "api.thingspeak.com";
String thingtweetAPIKey = "THINGTWEET_APIKEY";  // ThingTweet API key
 
SoftwareSerial softSerial(2, 4); // RX, TX
WiFiClient client;
 
int stat;
const int STAT_STOP = 0;
const int STAT_CLEAN = 1;
 
const int ROI_CMD_START = 128;
const int ROI_CMD_QUERY_LIST = 149;
const int ROI_PACKET_OIMODE = 35; // 1byte, unsigned
const int ROI_PACKET_CHARGE_SOURCE = 34;  // 1byte, unsigned
const int ROI_PACKET_BATTERY = 22;  // 2byte, unsigned
const int ROI_PACKET_TEMP = 24;  // 1byte, unsigned
const int ROI_PACKET_DISTANCE = 19;  // 2byte, signed
const int ROI_PACKET_MAIN_BRUSH_MOTOR = 56;  // 2byte, signed
const int ROI_PACKET_SIDE_BRUSH_MOTOR = 57;  // 2byte, signed
 
const int MSG_ALL = 0;
const int MSG_TWITTER_ONLY = 1;
const int MSG_DEBUG_ONLY = 2;
 
const int PERIOD = 1000;
const int MOTOR_CURRENT_TH = 10;
 
const int PIO_LED = 12;
const int PIO_SW = 13;
 
unsigned long last = 0;
unsigned long tick = 0;
unsigned long time_start = 0;
 
unsigned long traveled_mm;
 
void setup() {
  Serial.begin(115200);  // for Roomba
  softSerial.begin(115200);  // for Debug
 
  pinMode(PIO_LED, OUTPUT);
  pinMode(PIO_SW, INPUT);
  pinMode(14, OUTPUT);
  pinMode(16, OUTPUT);
  pinMode(5, OUTPUT);
   
  softSerial.println("Booting");
  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, password);
  while (WiFi.waitForConnectResult() != WL_CONNECTED) {
    softSerial.println("Connection Failed! Rebooting...");
    delay(5000);
    ESP.restart();
  }
 
  // Port defaults to 8266
  // ArduinoOTA.setPort(8266);
 
  // Hostname defaults to esp8266-[ChipID]
  // ArduinoOTA.setHostname("myesp8266");
 
  // No authentication by default
  // ArduinoOTA.setPassword((const char *)"123");
 
  ArduinoOTA.onStart([]() {
    softSerial.println("Start");
  });
  ArduinoOTA.onEnd([]() {
    softSerial.println("\nEnd");
  });
  ArduinoOTA.onProgress([](unsigned int progress, unsigned int total) {
    softSerial.printf("Progress: %u%%\r", (progress / (total / 100)));
  });
  ArduinoOTA.onError([](ota_error_t error) {
    softSerial.printf("Error[%u]: ", error);
    if (error == OTA_AUTH_ERROR) softSerial.println("Auth Failed");
    else if (error == OTA_BEGIN_ERROR) softSerial.println("Begin Failed");
    else if (error == OTA_CONNECT_ERROR) softSerial.println("Connect Failed");
    else if (error == OTA_RECEIVE_ERROR) softSerial.println("Receive Failed");
    else if (error == OTA_END_ERROR) softSerial.println("End Failed");
  });
  ArduinoOTA.begin();
   
  softSerial.println("Ready");
  softSerial.print("IP address: ");
  softSerial.println(WiFi.localIP());
 
  stat = STAT_STOP;
  last = millis();
  digitalWrite(PIO_LED, LOW);
}
 
void loop() {  
  ArduinoOTA.handle();
 
  tick = millis();
  if((tick - last) > PERIOD)
  {
    last = tick;
     
    if(Serial.available() != 11)
    {
      outputMsg("Only " + String(Serial.available()) + "bytes received.", MSG_DEBUG_ONLY);
      digitalWrite(PIO_LED, LOW);
    }
    else
    {
      byte rbuffer[16];
      for(int count = 0; count < 11; count++)
      {
        rbuffer[count] = Serial.read();
      }
 
      int oiMode = rbuffer[0];
      int chargeSource = rbuffer[1];
      int batteryVoltage = (rbuffer[2] << 8) | rbuffer[3];
      int temp = rbuffer[4];
      int dist = (rbuffer[5] << 8) | rbuffer[6];
      if(dist > 32767) dist -= 65536;
      traveled_mm += abs(dist);
      int mCurrent = (rbuffer[7] << 8) | rbuffer[8];
      if(mCurrent > 32767) mCurrent -= 65536;
      int sCurrent = (rbuffer[9] << 8) | rbuffer[10];
      if(sCurrent > 32767) sCurrent -= 65536;
 
      outputMsg("OIMode: " + String(oiMode), MSG_DEBUG_ONLY);
      outputMsg("ChargeSource: " + String(chargeSource), MSG_DEBUG_ONLY);
      outputMsg("Battery: " + String(batteryVoltage), MSG_DEBUG_ONLY);
      outputMsg("Temperature: " + String(temp), MSG_DEBUG_ONLY);
      outputMsg("Distance: " + String(dist), MSG_DEBUG_ONLY);
      outputMsg("Traveled: " + String(traveled_mm), MSG_DEBUG_ONLY);
      outputMsg("Main Brush Motor Current: " + String(mCurrent), MSG_DEBUG_ONLY);
      outputMsg("Side Brush Motor Current: " + String(sCurrent), MSG_DEBUG_ONLY);
       
      if(stat == STAT_STOP)
      {
        if(mCurrent > MOTOR_CURRENT_TH)  // main brush motor works
        {
          stat = STAT_CLEAN;
          time_start = millis();
          traveled_mm = 0;
          outputMsg("I've started to clean. (Battery: " + String(batteryVoltage) + "mV, Temp: " + String(temp) + "degC) by #Roomba", MSG_ALL);
          digitalWrite(PIO_LED, HIGH);
        }
      }
      else if(stat == STAT_CLEAN)
      {
        if(mCurrent == 0)  // main brush motor works stops
        {
          stat = STAT_STOP;
          float time_elapsed = (millis() - time_start) / 60000.0;  //convert to minute
          float traveled = traveled_mm / 1000.0;  // convert to m
          outputMsg("I finished cleaning in " + String(time_elapsed) + "minutes. The distance traveled is " + String(traveled) + "m. (Battery: " + String(batteryVoltage) + "mV, Temp: " + String(temp) + "degC) by #Roomba", MSG_ALL);
          digitalWrite(PIO_LED, LOW);
        }
      }
    }
 
    while(Serial.available() != 0) Serial.read();
    byte wbuffer[] = {
      byte(ROI_CMD_START),
      byte(ROI_CMD_QUERY_LIST),
      byte(7),  // number of packets
      byte(ROI_PACKET_OIMODE),
      byte(ROI_PACKET_CHARGE_SOURCE),
      byte(ROI_PACKET_BATTERY),
      byte(ROI_PACKET_TEMP),
      byte(ROI_PACKET_DISTANCE),
      byte(ROI_PACKET_MAIN_BRUSH_MOTOR),
      byte(ROI_PACKET_SIDE_BRUSH_MOTOR)
    };
    Serial.write(wbuffer, 10);
  }
}
 
void outputMsg(String str, int type)
{
  if(type == MSG_ALL || type == MSG_TWITTER_ONLY)
  {
    updateTwitterStatus(str);
  }
  if(type == MSG_ALL || type == MSG_DEBUG_ONLY)
  {
    softSerial.println(str);
  }
}
 
void updateTwitterStatus(String tsData)
{
  if (client.connect(thingSpeakAddress, 80))
  { 
    softSerial.println("Connected to ThingSpeak.");
    softSerial.println("Posted:" + tsData);
 
    // Create HTTP POST Data
    tsData = "api_key="+thingtweetAPIKey+"&status="+tsData;
 
    client.print("POST /apps/thingtweet/1/statuses/update HTTP/1.1\n");
    client.print("Host: api.thingspeak.com\n");
    client.print("Connection: close\n");
    client.print("Content-Type: application/x-www-form-urlencoded\n");
    client.print("Content-Length: ");
    client.print(tsData.length());
    client.print("\n\n");
 
    client.print(tsData);
    client.stop();
  }
  else
  {
     softSerial.println("Connection failed.");
  }
}
